function [x,y,jj,errrel] = jacobi(n,tolerance,maxiter,ncheck)
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                        %
% METODO DE JACOBI APLICADO A UN CASO DE EJEMPLO                         %
%                                                                        % 
%  [x,y,jj,errrel] = jacobi(n,tolerance,maxiter,ncheck)                  % 
%                                                                        %
%  ENTRADA                                                               %
%                                                                        %
%          n    numero de puntos de la malla                             %
%  tolerance    variacion relativa objetivo entre iteraciones sucesivas  %
%    maxiter    maximo numero de iteraciones                             %
%     ncheck    numero de iteraciones tras las cuales se muestra la      %
%               representacion grafica de las iteracion                  %
%                                                                        %
%                                                                        %
%  SALIDA                                                                %
%                                                                        %
%         x     malla en X                                               %
%         y     solucion en los puntos de la malla                       %
%     niter     numero de iteraciones realizadas                         %
%    errrel     error relativo maximo respecto a la solucion analitica   %
%                                                                        %
%                                                                        %
% Consideramos la ecuacion y''-5y'+10y=10x con cc y(0)=0, y(1)=100,      %
% aproximada por diferencias finitas de segundo orden. Resolvemos el     %
% sistema de ecuaciones resultante iterando segun el metodo de JACOBI.   %
%                                                                        %
% Se compara la solucion con la solucion analitica. En el caso de que    %
% se disponga de la "Symbolic Math Toolbox" puede hallarse automatica-   %
% mente tal solucion analitica.                                          %
%                                                                        %
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% limpieza del espacio de trabajo
figure(11);clf; figure(12);clf; figure(13);clf; 

%
% PARAMETROS DEL PROBLEMA ORIGINAL
%
a = 0.0;       % limite inferior del dominio
b = 1.0;       % limites superior del dominio
               % CONDICIONES DE CONTORNO
ya = 0.0;      % valor de la funcion en el limite inferior del dominio
yb = 1.0e2;    % valor de la funcion en el limite superior del dominio

%
% PARAMETROS DEL PROBLEMA DISCRETIZADO
%
%n = 101;           % numero de puntos de la malla
%tolerance = 2e-9; % tolerancia relativa de la solucion
%maxiter = 40000;   % maximo numero permitido de iteraciones
%ncheck = 2500;      % revisamos la solucion cada ncheck iteraciones

% malla equiespaciada en variable indendiente
x = linspace( a, b, n); 
dx = x(2)-x(1);

% coeficientes de la ODE discretizada
c1 = 1.0 / (2.0 - 10.0*dx^2);
c2 = c1 * ( 1.0 - 2.5*dx);
c3 = c1 * ( 1.0 + 2.5*dx);
c4 = -c1*10.0*dx^2;


% PROPUESTA DE SOLUCION inicial: estimacion lineal.
y = linspace( ya, yb, n); 

% El metodo de Jacobi requiere de un array auxiliar analogo a y. Lo 
% llamamos yy. Declaramos yy mediante la siguiente linea.
yy = y;

% Comenzamos las iteraciones. Se itera hasta que o bien se alcanza 
% la tolerancia requerida,  o bien se alcanza el numero maximo de 
% iteraciones.

done = false; % parametro de control del bucle
jj = 0;       % contador de iteraciones

kk = 0;       % contador auxiliara para visualizacion
epskk = ones(1,maxiter/ncheck)*tolerance;

figure(11);    % figura para representar las soluciones 
hold on;      % intermedias

while done == false
    
    if( jj > maxiter ), error('Demasiadas iteraciones'), end;
    done = true;
    jj = jj + 1;
    
    % yy representa a la solucion en el paso j-1.
    % y  representa a la solucion en el paso j.
    
    yy = y;
    % Bucle en y_i. Se puede sustituir por una operacion de array.
%     for ii = 2:n-1
%         y( ii ) = c2*yy(ii+1) + c3*yy(ii-1) + c4*x( ii );
%     end
    y( 2: n-1) = c2*yy( 3: n) + c3*yy( 1: n-2) + c4*x( 2:n-1 );
   
    % calculamos la maxima diferencia entre una iteracion y la 
    % siguiente y la almacenamos en epsjj
    
    epsjj = max( abs( (yy(2:n-1)-y(2:n-1)) / yy(2:n-1) )); 
    if ( epsjj > tolerance )
        done = false;
    end
    
    % cada ncheck iteraciones guardamos el error relativo y dibujamos
    % la solucion (la funcion rem devuelve el resto de la division).
    if( rem( jj-1 , ncheck ) == 0)
        plot(x,y,'-','Color',[ (jj-1)/maxiter 0 0 ]);
        kk = kk+1;
        epskk(kk) = epsjj;
    end
end
xlabel('x'); ylabel('y');
legend('Soluciones intermedias-Jacobi','Location','North');
fprintf('\n Metodo de Jacobi');
fprintf('\n iteraciones = %d\n',jj);

figure(12);
semilogy( 1:ncheck:(kk*ncheck), epskk(1:kk),'o');
xlabel('numero de iteracion-Jacobi'); ylabel('error relativo');

% comparamos con la solucion analitica
ff = matlabFunction( dsolve( 'D2y - 5*Dy + 10*y = 10*x','y(0)=0', ...
        'y(1)=100','x') );
errrel = max( abs( (y-ff(x))/y ));
fprintf('\n error relativo del resultado %e\n', errrel);    
fprintf('\n tolerancia %e\n', tolerance);

% dibujamos la solucion aproximada por Jacobi y la analitica
figure(13);
plot(x,y,'ro',x,ff(x),'-b');
xlabel('x'); ylabel('y');
legend('Solucion por Jacobi','Solucion analitica','Location','North');


